extends Node

# This class is responsible of loading the games scenes 
# and loading the game pck 

enum ScenesName {
	GAME,
	HOME_UI,
	LVL_WIN,
	LVL_LOSE,
	SKINS,
	GIFT,
	ADS,
	NOTIFICATION,
	GDPR,
}


const GAME_SCENES := {
	ScenesName.GAME : "res://scenes/Game.tscn",
	ScenesName.HOME_UI : "res://scenes/ui/home.tscn",
	ScenesName.LVL_WIN : "res://scenes/ui/level_win.tscn",
	ScenesName.LVL_LOSE : "res://scenes/ui/level_lose.tscn",
	ScenesName.SKINS : "res://scenes/ui/skins.tscn",
	ScenesName.GIFT : "res://scenes/ui/gift.tscn",
	ScenesName.ADS : "res://scenes/framework/b4d_ads.tscn",
	ScenesName.NOTIFICATION : "res://scenes/prefabs/notification.tscn",
	ScenesName.GDPR : "res://scenes/framework/gdpr.tscn",
	
}

onready var _thread : Thread  = B4DFramework.main_thread
var _loader : ResourceInteractiveLoader
var _total_scenes
var _total_loaded = 0


onready var _tween : Tween = get_node("Tween")
onready var _progress_bar : ProgressBar = get_node("splash/loading/ProgressBar")

func _ready():
	_total_scenes = GAME_SCENES.size()
	_start_thread("_load_pack",2)
	
	Utils.add_admob_manifest(GameSettings.ADMOB_APP_ID)
	Utils.add_multidex()


func _start_thread(func_name : String , priority : int):
	while true:
		var err = _thread.start(self,func_name, null, priority)
		if err == OK:
			break
		yield(get_tree().create_timer(0.5),"timeout")



func _load_pack() -> bool:
	var is_loaded := ProjectSettings.load_resource_pack("res://game.pck")
	call_deferred("_load_pack_deferred")

	return is_loaded


func _load_pack_deferred():
	var is_loaded = _thread.wait_to_finish()
	Utils.debug("_load_pack_deferred",1,str(is_loaded))
	_start_thread("_load_game",1)


func _load_game() -> int:
	
	var err

	for scn_name in GAME_SCENES:
		_loader = ResourceLoader.load_interactive(GAME_SCENES[scn_name])
		while true:
			err = _loader.poll()
			if err == ERR_FILE_EOF: # Finished loading.
				_total_loaded += 1
				_scene_ref(scn_name,_loader.get_resource())
				_update_progress()
				_loader = null
				break
			elif err == OK:
				pass
			else: # Failed
				_loader = null
				break
	
	call_deferred("_load_game_deferred")
#	yield(get_tree(),"idle_frame")
	return _total_loaded


func _load_game_deferred() -> void :
	var total_scenes : int = _thread.wait_to_finish()
	Utils.debug("_load_game_deferred",1,str(total_scenes))


func _update_progress():
	var progress = (float(_total_loaded) * 100) / _total_scenes
	_tween.interpolate_property(_progress_bar,
	"value",
	_progress_bar.get_value(),
	progress,
	1,
	Tween.TRANS_EXPO,
	Tween.EASE_OUT)
	_tween.start()


# Save loaded scenes in a reference
func _scene_ref(name : int , scene : PackedScene) -> void:
	Utils.debug("_scene_ref : scene loaded " , 1 , str(name))
	match name:
		ScenesName.GAME:
			B4DFramework.game_scn = scene
		ScenesName.HOME_UI:
			B4DFramework.home_ui_scn = scene
		ScenesName.LVL_WIN:
			B4DFramework.lvl_win_scn = scene
		ScenesName.LVL_LOSE:
			B4DFramework.lvl_lose_scn = scene
		ScenesName.SKINS:
			B4DFramework.skins_scn = scene
		ScenesName.GIFT:
			B4DFramework.gift_scn = scene
		ScenesName.ADS:
			B4DFramework.ads_scn = scene
		ScenesName.NOTIFICATION:
			B4DFramework.pop_notif_scn = scene
		ScenesName.GDPR:
			B4DFramework.gdpr_scn = scene


func _start_game() -> void:
	if B4DFramework.is_new_user:
		get_tree().change_scene_to(B4DFramework.gdpr_scn)
	else:
		get_tree().change_scene_to(B4DFramework.game_scn)


func _on_tween_all_completed():
	if _progress_bar.value >= 100:
		_start_game()
